////////////////////////////////////////////////////
// nopeustesti.c
////////////////////////////////////////////////////
// Copyright 2013 - 2020 Jaakko Kairus
// jaakko@kair.us
//
// You may freely use and modify this software for personal use.
// Commercial use of this software or parts of this software
// is prohibited without written permission from author.
// DISCLAIMER: This software comes with ABSOLUTELY NO WARRANTY
//
// For more information about this project see project web page at
// http://kair.us/projects/nopeustesti/index.html
//
// Compiled with CCS compiler version 5.097
// Total compiled length 5566 words
///////////////////////////////////////////////////
// Version history
//
// 2.3.2013 initial version 0.1
//
// 30.9.2013 changes from 0.1 to 0.2
// - Changed duty cycle of green button to remove resonance of speaker
// - Changed startup demo text to show 0.2
// - Added difficult game version to yellow button - starts from 100 points
// - Added memory game to green button. Memory test has separate high score
//
// 2.10.2013 changes from 0.2 to 0.3
// - added parameter to saa1064 scrolling function to stop scrolling if button is pressed
// - moved displaying of high scores inside switch-case. now also red button show hiscores
// - fixed memory test bug of not clearing seq_length after game
//
// 16.3.2014 changes from 0.3 to 0.4
// - clearing newpress before displaying high score message
//
// 13.1.2019 changes from 0.4 to 0.5
// - Updated to PIC16F15355 and driving display directly from PIC I/O (not SAA1064 anymore)
//
// 9.2.2019 changes from 0.5 to 0.6
// - Updated to using a PCB designed specifically for Nopeustesti
// - Added setup mode, entered by keeping all buttons pressed during power on
// - Updated speed game speedup algorithm to better match original game
// - Updated game over sound to better match original game
// - Demo mode not clearing timer0 anymore so all possible pseudorandom sequences possible
// - Selection between old and new sound mode, also game over sound
// - Possible to clear high scores from setup menu
//
// 6.3.2019 changes from 0.6 to 0.7
// - Added sanity check for saved FLASH values / settings
// - Cleanup
//
// 26.3.2019 changes from 0.7 to 0.8
// - Added button contact bounce test mode (keep left- and rightmost buttons pressed)
//
// 22.7.2019 changes from 0.8 to 0.9
// - Added function convert_periods() which removes blank spaces before periods
//   in a scrolling message
//
// 7.9.2019 changes from 0.9 to 0.91
// - Changed button order to red,yellow,green,blue. This new placement
//   order uses the three most common colors for starting different game modes
//   so the front panel instructions are valid if even if rightmost button is
//   amber or white instead of blue.
//
// 13.9.2019 changes from 0.91 to 0.92
// - Added setup option for dimming button lights. High score clearing moved
//   to a separate hidden boot-up option, by pressing two centre buttons.
//
// 28.11.2020 changes from 0.92 to 0.93
// - Updated compiler to 5.097
// - Automated output file name generation
// - Fixed press debounce (timer was not stopped if button was held down)
// - Some improvements to contact bounce test mode
// - Changed 'blue' to 'last' in startup texts, so they are valid also if
//   the rightmost button is orange or white.


#include <nopeustesti.h>

#int_timer1
void timer1isr(void) {
   set_timer1(64536);   // 1000 cycles before overflow - sets interrupt rate to approx 1 kHz

   // Seven segment display multiplexing
   output_a(_SPACE); // clear anodes
   switch(CURRENT_SEG) {
      case 0:   SEGMENT_A
            output_a(((DISPLAY[0]&0x01)<<7)|((DISPLAY[1]&0x01)<<6)|((DISPLAY[2]&0x01)<<5)|((DISPLAY[3]&0x01)<<4));
            break;
      case 1:   SEGMENT_B
            output_a(((DISPLAY[0]&0x02)<<6)|((DISPLAY[1]&0x02)<<5)|((DISPLAY[2]&0x02)<<4)|((DISPLAY[3]&0x02)<<3));
            break;
      case 2:   SEGMENT_C
            output_a(((DISPLAY[0]&0x04)<<5)|((DISPLAY[1]&0x04)<<4)|((DISPLAY[2]&0x04)<<3)|((DISPLAY[3]&0x04)<<2));
            break;
      case 3:   SEGMENT_D
            output_a(((DISPLAY[0]&0x08)<<4)|((DISPLAY[1]&0x08)<<3)|((DISPLAY[2]&0x08)<<2)|((DISPLAY[3]&0x08)<<1));
            break;
      case 4:   SEGMENT_E
            output_a(((DISPLAY[0]&0x10)<<3)|((DISPLAY[1]&0x10)<<2)|((DISPLAY[2]&0x10)<<1)|((DISPLAY[3]&0x10)));
            break;
      case 5:   SEGMENT_F
            output_a(((DISPLAY[0]&0x20)<<2)|((DISPLAY[1]&0x20)<<1)|((DISPLAY[2]&0x20))|((DISPLAY[3]&0x20)>>1));
            break;
      case 6:   SEGMENT_G
            output_a(((DISPLAY[0]&0x40)<<1)|((DISPLAY[1]&0x40))|((DISPLAY[2]&0x40)>>1)|((DISPLAY[3]&0x40)>>2));
            break;
      case 7:   SEGMENT_DP
            output_a(((DISPLAY[0]&0x80))|((DISPLAY[1]&0x80)>>1)|((DISPLAY[2]&0x80)>>2)|((DISPLAY[3]&0x80)>>3));
            break;
      default:SEGMENT_NONE
   }
   
   CURRENT_SEG++;
   if (CURRENT_SEG==8)
      CURRENT_SEG=0;
   
   read_buttons();
   
   if (CURRENT_SEG < settings_st.dimmer_mode)      // lamps are now only controlled here in interrupt
      output_b(lamp);                              // based on the lamp variable
   else
      output_b(ALL_LAMPS_OFF);
}


void main(void) {
   int8 i=0;
   int16 viive=START_DELAY;
   int16 memgameseed=0;
   int8 seq_length=0;
   int8 memorybutton=0;
   unsigned int16 demo_delay;
   
   set_tris_a(0b00001111);
   set_tris_b(0b11000000);
   set_tris_c(0x00);
   set_tris_e(0b11111000);
   port_a_pullups(0b00001111);
   port_b_pullups(0b11000000);
   port_e_pullups(0b00001000);
   output_a(0x00);
   output_b(0x00);
   output_c(0x00);

   // read high scores and settings from HEF flash 
   read_program_memory8(HEF_BASE, &settings_st, sizeof(settings_st));
   
   // sanity check for saved values / settings
   if (settings_st.hiscore < 0 || settings_st.hiscore > 9999)
      settings_st.hiscore = 0;
   if (settings_st.memhiscore < 0 || settings_st.memhiscore > 9999)
      settings_st.memhiscore = 0;
   if (settings_st.speaker_mode > SPEAKER_OFF)
      settings_st.speaker_mode = SPEAKER_QUIET;
   if (settings_st.sound_mode > SOUND_MEMORY)
      settings_st.sound_mode = SOUND_NEW;
   if (settings_st.demo_mode > 1)
      settings_st.demo_mode = 0;
   if (settings_st.dimmer_mode > 8)
      settings_st.dimmer_mode = 8;
  
   show_7seg_string(" ");

   setup_timer_1(T1_INTERNAL|T1_DIV_BY_1);   // INTERNAL means Fosc/4. With 4 MHz Fosc, these settings mean 65.536 ms period
   enable_interrupts(INT_TIMER1);
   enable_interrupts(GLOBAL);

   setup_timer_0(RTCC_INTERNAL|RTCC_DIV_64); // set timer0 to maximum of 4s
                                             // INTERNAL means Fosc/4. Prescaler 64. So timer0 increments every 64 s.

   #ifdef ALWAYS_DEBOUNCE_TEST     // Used only in special test equipment, not in actual speed test!
   bounce_test();
   #endif

   if ((input_a()&0x0f)==0x00)     // If all buttons are pressed during power on, enter setup menu
      setup_menu();
  
   if ((input_a()&0x09)==0x00)     // If left- and rightmost buttons are pressed during power on, enter contact bounce test mode
      bounce_test();

   if ((input_a()&0x06)==0x00)     // If two centre buttons are pressed during power on, ask for clearing of high scores
      clear_scores();
  
   new_press = 0;
   show_7seg_scroll(STARTUP_MESSAGE,DEFAULT_SCROLL_SPEED, &new_press);
   
   while(TRUE) {
      switch (mode) {      // DEMO
       case ST_DEMO:
         pin_select("NULL",BUZZER_P);   // disable buzzer for demo mode
         pin_select("NULL",BUZZER_SPK_N);

         if (new_press) {
            alloff();
            mode=ST_SPEED_GAME;
            srand(get_timer0());
            if (current_button==BUT_YELLOW)   {   // yellow button
               pisteet = 100;         // difficult game - start from 100
               for (i=0; i<pisteet; i++)
                  viive = decrease_delay(viive);
               //show_7seg_num(viive);     // uncomment these two lines to show
               //delay_ms(5000);           // delay in timer ticks (64s per tick)
            }      
            if (current_button==BUT_GREEN)   {   // green button
               mode=ST_MEMORY_GAME;               // memory game
               memgameseed = get_timer0(); // get 16 bit seed for memory game
            }      
            if (current_button==BUT_BLUE)      // blue (or last) button
               mode=ST_DISPLAY_HIGH_SCORES;               // show high scores
                  
            if (mode != ST_DISPLAY_HIGH_SCORES) {         // do not show 0 and wait one second if hiscore disp
               show_7seg_num(pisteet);
               delay_ms(1000);
            }   
            pwm_on();         // enable buzzer output for game
            pin_select("CWG1OUTA",BUZZER_P);
            pin_select("CWG1OUTB",BUZZER_SPK_N);

            new_press=0;
            set_timer0(0);
         }

         if (settings_st.demo_mode) {     // If demo mode active, blink buttons randomly
            demo_delay++;
            if (demo_delay==10000) {
               litnext(1);
               demo_delay=0;
            }
            if (demo_delay==8000)
               alloff();
         }
         break;
       
       case ST_SPEED_GAME:   // GAME
         if (new_press) {
            // if (1) {
            if (check_press()) {
               pisteet++;
               show_7seg_num(pisteet);
            } else
               mode=ST_GAME_OVER;
            new_press=0;
         }
            
         if (get_timer0()>viive) {
            litnext(1);
            //pisteet++;                  // uncomment these two lines and
            //show_7seg_num(pisteet);     // comment the following two to test speed-up speed
            if (addtostack(next)==0)
               mode=ST_GAME_OVER;
            set_timer0(0);
         viive = decrease_delay(viive);
         }
         
         if (get_timer0()>viive*2/3) {
            alloff();
         }
         
         if (mode == ST_GAME_OVER && pisteet>settings_st.hiscore)
            mode=ST_HIGH_SCORE;
         break;


       case ST_MEMORY_GAME:   // MEMORY GAME
          memorygame = TRUE;
          srand(memgameseed);
          for (i=0; i<=seq_length; i++) {
            litnext(0);
             delay_ms(250);
             alloff();
             delay_ms(105);
         }
         
         srand(memgameseed);
         for (i=0; i<=seq_length; i++) {
            while (!new_press) {
            }
            memorybutton = (rand()%4);
            new_press=0;
            if (current_button == memorybutton) {
               pisteet++;
               litbutton(memorybutton);
               show_7seg_num(pisteet);
               delay_ms(150);
               alloff();
            } else {
               if (pisteet>settings_st.memhiscore)
                  mode=ST_HIGH_SCORE;
               else
                  mode=ST_GAME_OVER;
               break;
            }   
         }
         if (mode == ST_MEMORY_GAME) { // Indicate correct sequence end by repeating last button
            litbutton(memorybutton);
            delay_ms(150);
            alloff();
            delay_ms(100);
            litbutton(memorybutton);
            delay_ms(250);
            alloff();
            delay_ms(1000);
         }
         seq_length++;      
         break;
      case ST_GAME_OVER:      //GAME OVER
         if (settings_st.sound_mode == SOUND_OLD) {
               lamp = ALL_LAMPS_ON;
            for (i=0; i<70; i++) {
               pwm_set_frequency(337);
               pwm_set_duty_percent(500);
               delay_ms(65);
               pwm_set_duty_percent(0);
               delay_ms(5);
            }
            alloff();
            for (i=0; i<70; i++) {
               pwm_set_frequency(580);
               pwm_set_duty_percent(500);
               delay_ms(65);
               pwm_set_duty_percent(0);
               delay_ms(5);
            }
         }
         else {
            alloff();
            for (i=0; i<4; i++) {
               pwm_set_frequency(490);
               pwm_set_duty_percent(500);
               delay_ms(400);
               pwm_set_frequency(325);
               pwm_set_duty_percent(500);
               delay_ms(400);
            }
            delay_ms(1200);   // Extends the last sound to 1.6 s
         }
         mode = ST_INIT;
         break;

      case ST_HIGH_SCORE:      // HIGH SCORE
         if (memorygame) {
            settings_st.memhiscore=pisteet;
         }
         else {   
            settings_st.hiscore=pisteet;
         }
         write_program_memory8(HEF_BASE, &settings_st, sizeof(settings_st));
         for (i=0; i<4; i++) {
            pwm_set_frequency(570);
            pwm_set_duty_percent(500);
            show_7seg_num(pisteet);
            lamp = ALL_LAMPS_ON;
            delay_ms(300);
            pwm_set_frequency(588);
            pwm_set_duty_percent(500);
            delay_ms(300);
            alloff();
            show_7seg_string(" ");
            pwm_set_frequency(606);
            pwm_set_duty_percent(500);
            delay_ms(300);
         }   
         alloff();
         new_press=0;
         show_7seg_scroll("congratulations new high score",DEFAULT_SCROLL_SPEED, &new_press);
         show_7seg_num(pisteet);
         mode = ST_INIT;
         break;         

       case ST_INIT:   // INIT
         new_press=0;
         pisteet=0;
         viive=START_DELAY;
         stacklevel=0;
         mode=ST_DEMO;
         memorygame=FALSE;
         seq_length=0;
         break;

       case ST_DISPLAY_HIGH_SCORES:   // DISPLAY HIGH SCORES DEMO
         show_7seg_scroll("speed test high score",DEFAULT_SCROLL_SPEED, &new_press);
         show_7seg_num(settings_st.hiscore);
         if (new_press == 0)
            delay_ms(1000);
         show_7seg_scroll("memory test high score",DEFAULT_SCROLL_SPEED, &new_press);
         show_7seg_num(settings_st.memhiscore);
         if (new_press == 0)
            delay_ms(1000);
         show_7seg_scroll("hit red button to start easy speed test",DEFAULT_SCROLL_SPEED, &new_press);
         show_7seg_scroll("hit yellow button to start hard speed test",DEFAULT_SCROLL_SPEED, &new_press);
         show_7seg_scroll("hit green button to start memory test",DEFAULT_SCROLL_SPEED, &new_press);
         show_7seg_scroll("hit last button to display high scores",DEFAULT_SCROLL_SPEED, &new_press);
         mode=ST_DEMO;
         break;

      }   // END OF SWITCH
   }   // END OF ENDLESS LOOP
}   // END OF MAIN

int check_press(void) {
   int i=0;
   if (stacklevel>0) {
      if (button_stack[0] == current_button) {
         stacklevel--;
         for (i=0;i<stacklevel;i++)
            button_stack[i] = button_stack[i+1];
         return 1;
      } else
         return 0;
   } else   
      return 0;
}      

int addtostack(int button) {
   if (stacklevel<MAXSTACK-1) {
      stacklevel++;
      button_stack[stacklevel-1] = next; // was button, not next
      return 1;
   } else
      return 0;   // stack is full
}      

void alloff(void) {
   lamp = ALL_LAMPS_OFF;
   pwm_set_duty_percent(0);
}

void litnext(int allnew) {
      next=(rand()%4);
      if (allnew) {
         while(next==previous)
            next=(rand()%4);
         previous=next;
      }
      litbutton(next);
}

void litbutton(int color) {
   unsigned int8 temp;
   temp = 0x01 << color;
   #ifdef ULN2003_DRIVE
   lamp = temp;
   #else
   lamp = (!temp) & 0x0f;      // Fiksaa - tss kuuluisi varmaan olla ~temp eik !temp
   #endif
   set_speaker_mode();
   pwm_set_frequency(button_frequency[color][settings_st.sound_mode]);
   pwm_set_duty_percent(500);
}

// Using 32 bit integer math
int16 decrease_delay(int16 viive) {
   if (viive>6230)
      viive=(int32)viive*993/1000;
   else if (viive>5100)
      viive=(int32)viive*996/1000;
   else if (viive>3000)
      viive=(int32)viive*9985/10000;
   else
      viive--;
   return viive;
}      

void show_7seg_num(signed int16 num)
{
   signed int8 j = 0;                       // Digit index
   signed int16 temp16;

   temp16 = labs(num);
   DISPLAY[j] = hex_table[temp16 %10];
   while (j < 3)
   {
      j++;
      if ((temp16/10) == 0 && num>=0)
         DISPLAY[j] = _SPACE;
      else if ((temp16/10) == 0 && temp16 != 0 && num < 0)
         DISPLAY[j] = _MINUS;
      else if (temp16 == 0 && num < 0)
         DISPLAY[j] = _SPACE;
      else
         DISPLAY[j] = hex_table[(temp16/10) %10];
      temp16 /= 10;
   }
}


// This function combines periods in a string with preceeding character. Avoids
// blank spaces between period and preceeding character
void convert_periods(char* message) {
   int8 i,l=strlen(message),periods=0;
   for (i=0;i<l-periods;i++) {
      message[i]=message[i+periods];
      if (message[i+periods+1] == '.') {
         periods++;
         message[i] += 128;
      }
   }
   message[i] = 0x00;   // Add null, thus shortening the string the amount of periods
}

//Sends scrolling string to led display buffer
void show_7seg_scroll(char* message,int8 speed,int8* new_press)
{
   convert_periods(message);
   
   int8 i,j;
   int8 buffer[5];
   for (i=0;i<strlen(message)+6;i++) {
      if (*new_press != 0)
         return;
      for (j=0;j<4;j++) {
         if (i+j<5||i+j>strlen(message)+4) // blanks before and after message
            buffer[j]=0x20;   // ascii space
         else
            buffer[j] = message[i+j-5];
      }
      buffer[4]=0x00;   // ensure the string is null terminated
      show_7seg_string(buffer);   
      delay_ms(speed);
   }
}

//Sends static string to led display buffer
void show_7seg_string(char* message)
{
   int8 k,i=strlen(message)-1;
   char temp;
   for (int j=0;j<4;j++) {
      k=i-j;
      temp = toupper(message[k]&0x7f); 
      if (j>i)
         DISPLAY[j]=_SPACE;
      else  {// (hopefully) a letter
         DISPLAY[j]=ascii_table[temp-' '];
         if (message[k] > 127)
            DISPLAY[j] = DISPLAY[j] _ADDPOINT;
      }
   }
}

void setup_menu(void)
{
   while ((input_a()&0x0f) != 0x0f) {}    // wait that all buttons are released
   delay_ms(100);
   new_press = 0;
   lamp = ALL_LAMPS_ON;   // Keep all lamps/leds on while in setup menu
                  // This causes the power to be switced off quicker
                  // and gives visual sign when actually switched off
   show_7seg_scroll("setup menu", DEFAULT_SCROLL_SPEED, &new_press);
   
   while(TRUE) {
      if (new_press) {
            new_press = 0;
            switch (current_button) {
               case BUT_RED:
                  settings_st.speaker_mode++;
                  if (settings_st.speaker_mode > SPEAKER_OFF)
                     settings_st.speaker_mode = SPEAKER_LOUD;
                  write_program_memory8(HEF_BASE, &settings_st, sizeof(settings_st));     // Write the changed (all) settings to HEF FLASH
                  switch (settings_st.speaker_mode) {
                     case SPEAKER_LOUD:
                        show_7seg_scroll("loud", DEFAULT_SCROLL_SPEED, &new_press);
                        break;
                     case SPEAKER_QUIET:
                        show_7seg_scroll("quiet", DEFAULT_SCROLL_SPEED, &new_press);
                        break;
                     case SPEAKER_EXTERNAL:
                        show_7seg_scroll("external", DEFAULT_SCROLL_SPEED, &new_press);
                        break;
                     case SPEAKER_OFF:
                        show_7seg_scroll("sound off", DEFAULT_SCROLL_SPEED, &new_press);
                        break;
                  }
                  break;
               case BUT_YELLOW:
                  settings_st.sound_mode++;
                  if (settings_st.sound_mode > SOUND_MEMORY)
                     settings_st.sound_mode = SOUND_OLD;
                  write_program_memory8(HEF_BASE, &settings_st, sizeof(settings_st));     // Write the changed (all) settings to HEF FLASH
                  switch(settings_st.sound_mode) {
                     case SOUND_OLD:
                        show_7seg_scroll("old sound", DEFAULT_SCROLL_SPEED, &new_press);
                        break;
                     case SOUND_NEW:
                        show_7seg_scroll("new sound", DEFAULT_SCROLL_SPEED, &new_press);
                        break;
                     case SOUND_MEMORY:
                        show_7seg_scroll("memory game sound", DEFAULT_SCROLL_SPEED, &new_press);
                        break;
                  }
                  break;
               case BUT_GREEN:
                  if (settings_st.demo_mode) {
                     settings_st.demo_mode = 0;
                     write_program_memory8(HEF_BASE, &settings_st, sizeof(settings_st));     // Write the changed (all) settings to HEF FLASH
                     show_7seg_scroll("demo off", DEFAULT_SCROLL_SPEED, &new_press);
                  }
                  else {
                     settings_st.demo_mode = 1;
                     write_program_memory8(HEF_BASE, &settings_st, sizeof(settings_st));     // Write the changed (all) settings to HEF FLASH
                     show_7seg_scroll("demo on", DEFAULT_SCROLL_SPEED, &new_press);
                  }
                  break;
               case BUT_BLUE:
                  settings_st.dimmer_mode>>=1;
                  if (settings_st.dimmer_mode == 0)
                     settings_st.dimmer_mode = 0x08;
                  write_program_memory8(HEF_BASE, &settings_st, sizeof(settings_st));     // Write the changed (all) settings to HEF FLASH
                  switch(settings_st.dimmer_mode) {
                     case 1:
                        show_7seg_scroll("dimmest", DEFAULT_SCROLL_SPEED, &new_press);
                        break;
                     case 2:
                        show_7seg_scroll("dimmer", DEFAULT_SCROLL_SPEED, &new_press);
                        break;
                     case 4:
                        show_7seg_scroll("dim", DEFAULT_SCROLL_SPEED, &new_press);
                        break;
                     case 8:
                        show_7seg_scroll("bright", DEFAULT_SCROLL_SPEED, &new_press);
                        break;
                  }
                  break;
            }
// Moved the flash writes to preceed scroll messages so settings are still stored if power is cut before message finishes
//            write_program_memory8(HEF_BASE, &settings_st, sizeof(settings_st));     // Write the changed (all) settings to HEF FLASH
          }
      }
}

// There are several ways to disable buzzer. At least:
// - Set PWM duty cycle to zero
// - Disable CWG
// - Set PPS (peripheral pin select) PWM output pins to NULL instead of CWG1OUTA/CWG1OUTB
// - Use CWG_A_FORCE_LOW/CWG_B_FORCE_LOW options with setup_cwg
// - Use pwm_off()

void set_speaker_mode(void)
{
   switch (settings_st.speaker_mode) {
      case SPEAKER_OFF:
         setup_cwg(CWG_ENABLED|CWG_MODE_SYNCRONOUS_STEERING|CWG_A_FORCE_LOW|CWG_B_FORCE_LOW|CWG_INPUT_CCP1, CWG_NO_AUTO_SHUTDOWN|CWG_SHUTDOWN_AC_LOW|CWG_SHUTDOWN_BD_LOW, 0x00, 0x00);
         break;
      case SPEAKER_QUIET:
         setup_cwg(CWG_ENABLED|CWG_MODE_SYNCRONOUS_STEERING|CWG_A_OUTPUT_PWM|CWG_B_FORCE_LOW|CWG_INPUT_CCP1, CWG_NO_AUTO_SHUTDOWN|CWG_SHUTDOWN_AC_LOW|CWG_SHUTDOWN_BD_LOW, 0x00, 0x00);
         break;
      case SPEAKER_LOUD:
         setup_cwg(CWG_ENABLED|CWG_MODE_SYNCRONOUS_STEERING|CWG_A_OUTPUT_PWM|CWG_B_OUTPUT_PWM|CWG_B_INVERTED|CWG_INPUT_CCP1, CWG_NO_AUTO_SHUTDOWN|CWG_SHUTDOWN_AC_LOW|CWG_SHUTDOWN_BD_LOW, 0x00, 0x00);
         break;
      case SPEAKER_EXTERNAL:
         setup_cwg(CWG_ENABLED|CWG_MODE_SYNCRONOUS_STEERING|CWG_A_OUTPUT_PWM|CWG_B_OUTPUT_PWM|CWG_INPUT_CCP1, CWG_NO_AUTO_SHUTDOWN|CWG_SHUTDOWN_AC_LOW|CWG_SHUTDOWN_BD_LOW, 0x00, 0x00);
   }
}

void bounce_test(void) {
   unsigned int16 bounce_time;
   unsigned int16 release_display_delay;
   while ((input_a()&0x0f) != 0x0f) {}    // wait that all buttons are released
   delay_ms(100);
   new_press = 0;
   lamp = ALL_LAMPS_OFF;               // lamps off when in contact bounce test mode because
                              // disabling interupts causes button dimming to not work
                              // another solution would be to use full brightness
                              // in this mode
   show_7seg_scroll("contact bounce test mode", DEFAULT_SCROLL_SPEED, &new_press);

   setup_timer_0(RTCC_INTERNAL|RTCC_DIV_2); // set timer0 to maximum of 130 ms


   while(TRUE) {
   
      // Press contact bounce time
      disable_interrupts(INT_TIMER1);
      output_a(_SPACE); // clear anodes
      while((input_a()&0x0f) == 0x0f) {} // wait for a button to be pressed
      set_timer0(0);          // clear timer0
      bounce_time = 0;
      TMR0IF = 0;             // clear interrupt flag
      while(TMR0IF == 0) {        // run until timer0 overflows
         if((input_a()&0x0f) == 0x0f)     // capture timer0 reading when contact is open
            bounce_time = get_timer0();
      }
      show_7seg_num(bounce_time/50);    // bounce time in milliseconds, with one decimal
      if (display[1] == _SPACE)
         display[1] = _0;
      display[1] = display[1] _ADDPOINT;
      enable_interrupts(INT_TIMER1);
      
      // Release contact bounce time
      while ((input_a()&0x0f) != 0x0f) {}    // wait that all buttons are released
      set_timer0(0);          // clear timer0
      disable_interrupts(INT_TIMER1);
      output_a(_SPACE); // clear anodes
      bounce_time = 0;
      TMR0IF = 0;             // clear interrupt flag
      while(TMR0IF == 0) {        // run until timer0 overflows
         if((input_a()&0x0f) != 0x0f)     // capture timer0 reading when contact is closed
            bounce_time = get_timer0();
      }
      show_7seg_num(bounce_time/50);    // bounce time in milliseconds, with one decimal
      if (display[1] == _SPACE)
         display[1] = _0;
      display[1] = display[1] _ADDPOINT;
      display[3] = _R;
      enable_interrupts(INT_TIMER1);
      //delay_ms(50);
      
      release_display_delay=0;
      while (release_display_delay<45000 && ((input_a()&0x0f) == 0x0f)) {      // This shows the release time for ca. 0.5 seconds
         //delay_us(5);                                                     // but exits if a button is pressed. Thus allows
         release_display_delay++;                                          // quicker retesting of the button.
      }
   }
}

void read_buttons(void) {
   button_shadow = input_a() & 0x0f;
   // fiksaa debounce, katso j-ace! -- tai l kato, mutta muuta alla oleva rivi (but_shdw || tmr!=0) && tmr<PRESS_DEBOUNCE_TIME
   //if (button_shadow!=0x0f || (button_timer<PRESS_DEBOUNCE_TIME && button_timer!=0))   // Debounce on button press is defined by PRESS_DEBOUNCE_TIME
   // 29.11.2020 korjattu, mutta pitisik viel list tarkistus ett button_timer ei kynnisty uudestaan kun release_timer ei ole nolla?
   // itse asiassa tuo korjauskaan ei toimi, sill nyt button timer j pyrimn 0..PRESS_DEBOUNCE_TIME vli jos nappia pit pohjassa..
   // -> Listty tarkistusehto ennenkuin nollataan button_timer.
   // -> Listty mys ehto ett release timer pit olla nolla ennenkuin button_timer kynnistyy.
   // Toiminta testattava, voisi mys vertailla 0.92 fimrikseen saako nopeustestin testaajalla paremman tuloksen kun aiemmin.
   
   if (((button_shadow!=0x0f && release_timer==0) || button_timer!=0) && button_timer<PRESS_DEBOUNCE_TIME)   // Debounce on button press is defined by PRESS_DEBOUNCE_TIME
      button_timer++;
   else if (button_shadow==0x0f)
      button_timer=0;
   
   if (button_shadow==0x0f && release_timer)         // Release debounce is defined by RELEASE_DEBOUNCE_TIME
      release_timer--;
   else if (button_shadow!=0x0f && button_timer>1)
      release_timer=RELEASE_DEBOUNCE_TIME;
   
   if (button_timer == 1 && release_timer == 0 && new_press == 0) {
      if (!(button_shadow &0x08))
         current_button=BUT_RED;
      if (!(button_shadow &0x04))
         current_button=BUT_YELLOW;
      if (!(button_shadow &0x02))
         current_button=BUT_GREEN;
      if (!(button_shadow &0x01))
         current_button=BUT_BLUE;
      new_press = 1;
   }      
}

void clear_scores(void) {
   while ((input_a()&0x0f) != 0x0f) {}    // wait that all buttons are released
   delay_ms(100);
   new_press = 0;
   lamp = ALL_LAMPS_ON;
   show_7seg_scroll("clear high scores are you sure?  press red to confirm any to cancel", DEFAULT_SCROLL_SPEED, &new_press);
   while (!new_press) {};
   new_press=0;
   if (current_button == BUT_RED) {
      settings_st.hiscore = 0;
      settings_st.memhiscore = 0;
      write_program_memory8(HEF_BASE, &settings_st, sizeof(settings_st));     // Write the changed (all) settings to HEF FLASH
      show_7seg_scroll("high scores cleared", DEFAULT_SCROLL_SPEED, &new_press);
   }
   else {
      show_7seg_scroll("canceled", DEFAULT_SCROLL_SPEED, &new_press);
   }
   reset_cpu();
}
