////////////////////////////////////////////////////
// nopeustesti.h
////////////////////////////////////////////////////
// Copyright 2013 - 2020 Jaakko Kairus
// jaakko@kair.us
//
// You may freely use and modify this software for personal use.
// Commercial use of this software or parts of it is prohibited
// without written permission from author.
// DISCLAIMER: This software comes with ABSOLUTELY NO WARRANTY
//
// For more information about this project see project web page at
// http://kair.us/projects/nopeustesti/index.html
/////////////////////////////////////////////////////

#define FW_MAJOR 0
#define FW_MINOR 93

////////////////////////////////////////////////////////
/// GAME AND HARDWARE SETTINGS DEFINES
////////////////////////////////////////////////////////
#define START_DELAY 8906         // Delay (speed) at beginning of game in timer0 ticks (64 s per tick)
#define MAXSTACK 50              // maximum number of pending button presses. on unlimited genuine speed test this seems to be 50.
#define DEFAULT_SCROLL_SPEED 170 // message scroll speed. delay in milliseconds.
#define ULN2003_DRIVE            // PIC output is driven high to turn on lamp. If you want to control button leds directly by grounding cathode via PIC pin, comment out this line
#define PRESS_DEBOUNCE_TIME 15   // Button debounce time when pressing it down, in milliseconds
#define RELEASE_DEBOUNCE_TIME 5  // Button debounce time when releasing it, in milliseconds
// #define NOPEUSTESTI_CC_DISPLAY   // Uncomment this line if using common cathode display. Recommended is to use common anode display though.
// #define ALWAYS_DEBOUNCE_TEST     // When defined, always enters button debounce test at power on. Used in test equipment, do not use otherwise
//////////////////////////////////////////////////////
/// END OF GAME AND HARDWARE SETTINGS RELATED DEFINES
//////////////////////////////////////////////////////

#include <16F15355.h>      // change this to 16F15356.h to compile for that chip
#DEVICE PASS_STRINGS=IN_RAM // allows passing constant char arrays to show_7seg_scroll() and show_7seg_string() functions
                            // copies string from rom to ram when needed
#fuses NOPPS1WAY,NOMCLR
#use delay(internal=4MHz)
#pin_select CCP1OUT=PIN_B4
#use pwm(CCP1, TIMER=2, FREQUENCY=400, PWM_OFF)
#use FAST_IO(a)
#use FAST_IO(b)
#use FAST_IO(c)
#use FAST_IO(e)

// Helper macros for concatenating version numbers to string
#define STRINGIZE_NX(A) #A
#define STRINGIZE(A) STRINGIZE_NX(A)
#define VERCAT_NX(A, B, C) A ##B ##C
#define VERCAT(A, B, C) PPCAT_NX(A, B, C)	// this probably isn't needed anywhere..

#define FW_VER FW_MAJOR ## FW_MINOR                            // FW version for file name
#define FW_VER_S STRINGIZE(VERCAT_NX(FW_MAJOR,.,FW_MINOR))     // FW version as string for startup message

// Helper macros for export file name generation
#define CAT(a,b,c,d) a ##b ##c ##d
#define XCAT(a,b,c,d) CAT(a,b,c,d)

#if getenv("DEVICE")=="PIC16F15355"
   #define HEF_BASE 0x1FE0
   #define NT_PIC_TYPE _16F15355
#elif getenv("DEVICE")=="PIC16F15356"
   #define HEF_BASE 0x3FE0
   #define NT_PIC_TYPE _16F15356
#else
   #error Unknown device, can't reserve HEF memory
#endif

#ifdef ALWAYS_DEBOUNCE_TEST
   #define BASE_FILENAME debounce_test_v
#else
   #define BASE_FILENAME nopeustesti_v
#endif

#ifdef NOPEUSTESTI_CC_DISPLAY
   #export (HEX, FILE = XCAT(BASE_FILENAME,FW_VER,NT_PIC_TYPE,_cc.hex)) 
#else
   #export (HEX, FILE = XCAT(BASE_FILENAME,FW_VER,NT_PIC_TYPE,_ca.hex)) 
#endif

#include <stdlib.h>         // for random number generation

#byte INTF = 0x70C
#bit TMR0IF = INTF.5

#define STARTUP_STRING_1 "speden spelit nopeustesti versio "
#define STARTUP_MESSAGE STARTUP_STRING_1 FW_VER_S              // Combine the startup message string with the fw version string

#ifdef NOPEUSTESTI_CC_DISPLAY
#define SEGMENT_A output_c(0b00000001);
#define SEGMENT_B output_c(0b00000010);
#define SEGMENT_C output_c(0b00000100);
#define SEGMENT_D output_c(0b00001000);
#define SEGMENT_E output_c(0b00010000);
#define SEGMENT_F output_c(0b00100000);
#define SEGMENT_G output_c(0b01000000);
#define SEGMENT_DP output_c(0b10000000);
#define SEGMENT_NONE output_c(0b00000000);
#else
#define SEGMENT_A output_c(0b11111110);
#define SEGMENT_B output_c(0b11111101);
#define SEGMENT_C output_c(0b11111011);
#define SEGMENT_D output_c(0b11110111);
#define SEGMENT_E output_c(0b11101111);
#define SEGMENT_F output_c(0b11011111);
#define SEGMENT_G output_c(0b10111111);
#define SEGMENT_DP output_c(0b01111111);
#define SEGMENT_NONE output_c(0b11111111);
#define CC_DISPLAY   // Actually we use common anode display but as we mux
                     // segments and not digits we must use inverted patterns!
#endif

#include "7seg_chars.h"

#ifdef ULN2003_DRIVE
#define LAMP_ON 1
#define LAMP_OFF 0
#define ALL_LAMPS_ON 0x0f
#define ALL_LAMPS_OFF 0x00
#else
#define LAMP_ON 0
#define LAMP_OFF 1
#define ALL_LAMPS_ON 0x00
#define ALL_LAMPS_OFF 0x0f
#endif

#define BUZZER_P PIN_B4
#define BUZZER_SPK_N PIN_B5

int8 const hex_table[16]= {_0,_1,_2,_3,_4,_5,_6,_7,_8,_9,_A,_B,_C,_D,_E,_F};
int8 const ascii_table[64] =      {_SPACE,_EXCLAMATION,_QUOTATION,_HASH,_DOLLAR,_PERCENT,_AMPERSAND,_APOSTROPHE,
                           _LEFT_PARENTHESIS,_RIGHT_PARENTHESIS,_ASTERISK,_PLUS,_COMMA,_MINUS,_PERIOD,_SLASH,
                           _0,_1,_2,_3,_4,_5,_6,_7,
                           _8,_9,_COLON,_SEMICOLON,_LESS,_EQUALS,_GREATER,_QUESTION,
                           _AT,_A,_B,_C,_D,_E,_F,_G,
                           _H,_I,_J,_K,_L,_M,_N,_O,
                           _P,_Q,_R,_S,_T,_U,_V,_W,
                           _X,_Y,_Z,_LEFT_BRACKET,_BACKSLASH,_RIGHT_BRACKET,_CIRCUMFLEX,_UNDERSCORE};

unsigned int16 const button_frequency[4][3] = {
   // Old   // New   // Memory
   368,     122,     770,
   488,     163,     1000,
   588,     195,     1290,
   733,     244,     1686
};

// Init and reserve last program memory page for highscores and settings
// One page (row) is 32 14-bit program memory words. We only init 16 first
// words of last row since we don't need more storage space (currently only
// 7 words, size of settings_st struct. Only low byte of prg mem word is used)
// Only low byte of the last 128 prg mem words is HEF(!)
#rom HEF_BASE = {0x0000, 0x0000, 0x0000, 0x0000, 0x0001, 0x0001, 0x0000, 0x0008,
               0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000}

/////////////////////////////
// Global variables
////////////////////////////

unsigned int8 DISPLAY[4];         // Display memory
unsigned int8 CURRENT_SEG = 0;

signed int16 pisteet=0;
int8 next=0;
int8 previous = 0;
int8 button_stack[MAXSTACK];
int8 stacklevel = 0;
int8 button_shadow = 0;
int8 button_timer = 0;
int8 release_timer = 0;
int8 new_press=0;
int8 demo_mode=0;
int8 lamp=0;
#bit LAMP_RED = lamp.0
#bit LAMP_YELLOW = lamp.1
#bit LAMP_GREEN = lamp.2
#bit LAMP_BLUE = lamp.3
int1 memorygame=FALSE;

typedef enum {
   BUT_RED = 0,
   BUT_YELLOW = 1,
   BUT_GREEN = 2,
   BUT_BLUE = 3,
   BUT_NONE
} BUTTONS;
BUTTONS current_button = BUT_NONE;

typedef enum {
   SPEAKER_LOUD,
   SPEAKER_QUIET,
   SPEAKER_EXTERNAL,
   SPEAKER_OFF
} SPEAKER;

typedef enum {
   SOUND_OLD = 0,
   SOUND_NEW,
   SOUND_MEMORY,
   SOUND_OTHER
} SOUND;

struct game_settings {
   signed int16 hiscore;
   signed int16 memhiscore;
   SPEAKER speaker_mode;
   SOUND sound_mode;
   unsigned int8 demo_mode;
   unsigned int8 dimmer_mode;
};

struct game_settings settings_st;

typedef enum {
   ST_DEMO,
   ST_SPEED_GAME,
   ST_MEMORY_GAME,
   ST_GAME_OVER,
   ST_HIGH_SCORE,
   ST_DISPLAY_HIGH_SCORES,
   ST_INIT
} STATES;
static STATES mode = ST_DISPLAY_HIGH_SCORES;

//////////////////////////
// Function prototypes
//////////////////////////

void red(void);
void yellow(void);
void green(void);
void blue(void);
void alloff(void);
void litnext(int allnew);
void litbutton(int color);
int16 decrease_delay(int16 viive);
int check_press(void);
int addtostack(int);
void show_7seg_num(signed int16 num);
void convert_periods(char* message);
void show_7seg_scroll(char* message,int8 speed,int8* new_press);
void show_7seg_string(char* message);
void setup_menu(void);
void set_speaker_mode(void);
void bounce_test(void);
void read_buttons(void);
void clear_scores(void);


